<?php

namespace App\Http\Controllers\Api\V100\Auth;

use App\Helpers\Constant;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\V100\Auth\AuthenticationRequest;
use App\Http\Requests\Api\V100\Auth\SendUserConfirmationCodeRequest;
use App\Http\Requests\Api\V100\Auth\VerifyUserConfirmationCodeRequest;
use App\Http\Requests\Api\V100\Auth\SocialLoginUserRequest;
use App\Http\Resources\User\UserResource;
use App\Models\User;
use App\Models\VerificationCode;
use App\Services\Auth\VerificationService;
use App\Services\RespondActive;

class AuthController extends Controller
{
    public function authentication(AuthenticationRequest $request)
    {
        $user = User::checkUserExist($request->phone)->first();

        if (!$user) {
            $user = User::create($request->validated() + [
                'account_type'  => Constant::USER_TYPE['User'],
            ]);
        }

        if (auth('sanctum')->user()) {
            $user = auth('sanctum')->user();

            $user->update($request->validated());
        }

        $user->update(['verified'   => Constant::USER_STATUS['Not verified']]);

        VerificationService::verifyAccount(
            $user->id,
            Constant::VERIFICATION_OBJECTIVE['Verify'],
            Constant::VERIFICATION_INFORMATION_TYPE['Phone'],
            $request->phone
        );

        return RespondActive::success('Code sent to your phone, activate the account.');
    }

    public function socialLogin(SocialLoginUserRequest $request)
    {
        $user = User::whereSocialMediaProviderId($request->provider_id)->orWhere('phone', $request->phone)->first();

        $message = 'Logged in Successfully';

        if($request->phone && $user) {
            $user->update([
                'verified'                  => Constant::USER_STATUS['Verified'],
                'social_media'              => true,
                'social_media_objective'    => $request->objective,
                'name'                      => $request->name,
                'email'                     => $request->email,
                'phone'                     => $request->phone,
                'social_media_provider_id'  => $request->provider_id,
            ]);
        }

        if (!$user){
            $user = User::create([
                'verified'                  => Constant::USER_STATUS['Not verified'],
                'social_media'              => true,
                'social_media_objective'    => $request->objective,
                'name'                      => $request->name,
                'email'                     => $request->email,
                'phone'                     => $request->phone,
                'social_media_provider_id'  => $request->provider_id,
            ]);

            $message = 'Registered Successfully';
        }

        $user['token'] = $user->createToken("token".$user->id)->plainTextToken;

        return RespondActive::success($message, new UserResource($user));
    }

    public function sendCode(SendUserConfirmationCodeRequest $request)
    {
        $user = User::checkUserExist($request->phone)->first(['id']);

        VerificationService::verifyAccount(
            $user->id,
            Constant::VERIFICATION_OBJECTIVE['Verify'],
            Constant::VERIFICATION_INFORMATION_TYPE['Phone'],
            $request->phone
        );

        return RespondActive::success('Code sent successfully.');
    }

    public function verifyCode(VerifyUserConfirmationCodeRequest $request)
    {
        $check = VerificationCode::checkCode($request->phone, $request->code, Constant::VERIFICATION_OBJECTIVE['Verify'])->first();

        if (!$check) {
            return RespondActive::clientError('Invalid code.');
        }

        $user = User::checkUserExist($request->phone)->first();

        $user->update([ 'verified' => Constant::USER_STATUS['Verified'] ]);

        $check->update([ 'used' => Constant::VERIFICATION_USED['Used'] ]);

        $user['token'] = $user->createToken('token'.$user->id)->plainTextToken;

        return RespondActive::success('Verified successfully.', new UserResource($user));
    }

    public function logout()
    {
        auth()->user()->tokens()->delete();

        return RespondActive::success('Logged out successfully.');
    }
}
